<?php
/*
 * chouchenn wonderslip
 * Copyright (c) 2004 olo <wonderslip at olivierl dot org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/**************************************************
 * Facade
 **************************************************/

define('ALE_BasePath', WS_BasePath);
include(ALE_BasePath.'lexer-shortcuts.php');
include(ALE_BasePath.'lexer-serial.php');

/**************************
 * Constants
 **************************/

/* Available output formats. */
define('WS_OUT_HTML', 0);
define('WS_OUT_XML', 1);
define('WS_OUT_WML', 2);

/* Data length checking status. */
define ('WS_LENGTH_OK', 0);
define ('WS_LENGTH_NOK_POST', 1);
define ('WS_LENGTH_NOK_WORD', 2);

include(WS_BasePath.'ws-util.php');
include(WS_BasePath.'ws-tokentypes.php');
include(WS_BasePath.'ws-visitors.php');
include(WS_BasePath.'ws-actions.php');
include(WS_BasePath.'ws-states.php');

/**************************
 * Default config
 **************************/

/* Default wiki base url. */
$WS_defaultWiki = '';

/* Other wikis base url (associative array). */
$WS_wikis = array();

/* Smileys url pattern (where %s stands for the smiley name). */
$WS_smileyUrl = 'http://jr@forum.hardware.fr/images/perso/%s.gif';

/* Wiki base url for smileys. */
$WS_smileyWiki = '';

/* Known boards url. */
$WS_boards = array();

/* Current board names set. */
$WS_myNames = array();

/* Allowed tags set. */
$WS_allowedTags = array_ident(array('b', 'i', 'u', 's', 'tt'));

/* Tag replacement schemes. */
$WS_tagReplacements = array(
	WS_OUT_HTML => array(
		'u' => array('<span class="underline">', '</span>'),
		's' => array('<span class="strike">', '</span>'),
	),
	WS_OUT_XML => array(
	),
	WS_OUT_WML => array(
		's' => array('<u>', '</u>'),
		'tt' => array('<small>', '</small>'),
	)
);

/* Supported protocols. */
$WS_protocolsPattern = 'http://|https://|ftp://';

/**************************
 * Configuration functions
 **************************/

/*
 * Defines wikis base url.
 * $defaultWiki : defalut wiki base url (empty string to disable wiki
 * support).
 * $wikis : other wikis base url, (name->url) array.
 */
function WS_setWiki($defaultWiki, $wikis = array())
{
	global $WS_defaultWiki, $WS_wikis;
	$WS_defaultWiki = $defaultWiki;
	$WS_wikis = $wikis;
}

/*
 * Defines smiley url.
 * $smileyUrl : smileys url pattern (where %s stands for the smiley name,
 * empty string to disable smileys support).
 */
function WS_setSmileyUrl($smileyUrl)
{
	global $WS_smileyUrl;
	$WS_smileyUrl = $smileyUrl;
}

/*
 * Defines wiki base url for smileys.
 * $smileyWiki : wiki base url (empty string to disable wiki support for
 * smileys).
 */
function WS_setSmileyWiki($smileyWiki)
{
	global $WS_smileyWiki;
	$WS_smileyWiki = $smileyWiki;
}

/*
 * Defines boards (for clock support).
 * $boards : boards url, (name->url) array.
 * $myNames : current board names list.
 */
function WS_setBoards($boards, $myNames = array())
{
	global $WS_boards, $WS_myNames;
	$WS_boards = $boards;
	$WS_myNames = array_ident($myNames);
}

/*
 * Adds an allowed tags.
 * $tagName : name of the tag to allow.
 */
function WS_addAllowedTag($tagName)
{
	global $WS_allowedTags;
	$WS_allowedTags[$tagName] = $tagName;
}

/*
 * Adds a tag replacement scheme.
 * $tagName : name of the tag to replace.
 * $beginTag : replacement string for the opening tag.
 * $endTag : replacement string for the closing tag.
 * $out : output affected by the replacement (-1 for all).
 */
function WS_addTagReplacement($tagName, $beginTag, $endTag, $out = -1)
{
	global $WS_tagReplacements;
	if ($out == -1)
	{
		for($i = 0; $i < count($WS_tagReplacements); $i++)
		{
			$WS_tagReplacements[$i][$tagName] = array($beginTag, $endTag);
		}
	} else
	{
		$WS_tagReplacements[$out][$tagName] = array($beginTag, $endTag);
	}
}

/*
 * Defines allowed protocols for url.
 * $protocols : protocols list.
 */
function WS_setAllowedProtocols($protocols)
{
	global $WS_protocolsPattern;
	function protocolQuote($string)
	{
		return preg_quote($string, '#');
	}
	$WS_protocolsPattern = join('|', array_map('protocolQuote', $protocols));
}

/*
 * Adds a state to the states associative array.
 */
function WS_addState($stateName, $state)
{
	global $WS_states;
	$WS_states[$stateName] = $state;
}

/*
 * Adds a transition in the head of the transitions sequence (just after the
 * transition to apply when the end of data is reached) of a non final state.
 */
function WS_addTransition($stateName, $transition)
{
	global $WS_states;
	$WS_states[$stateName]->addTransition($transition);
}

/**************************
 * Main functions
 **************************/

/*
 * Returns the list of tokens compounding whole data.
 * $data : string to analyse (tokenize).
 */
function &WS_getTokens(&$data)
{
	global $WS_T_End, $WS_states;
	$lexer =& new ALE_Lexer($data, $WS_states);
	return $lexer->readTokens($WS_T_End);
}

/*
 * Checks data and each word lengths.
 * $tokens : list of tokens.
 * $postMaxLength : data max length.
 * $wordMaxLength : word max length.
 */
function WS_checkLength(&$tokens, $postMaxLength, $wordMaxLength)
{
	$postLength = 0;
	$wordLength = 0;
	foreach($tokens as $token) {
		$length = strlen($token->relevantText());
		$postLength += $length;
		if ($token->tokenType->isSpace($token)) $wordLength = 0;
		else $wordLength += $length;
/*
		print("<p>");
		print("<tt>".$token->tokenType->name."</tt> : ");
		print("<i>".htmlspecialchars($token->text)."</i> ; ");
		//print("\$postLength = $postLength ; \$wordLength = $wordLength");
		print("</p>\n");
*/
		if ($postLength > $postMaxLength) return WS_LENGTH_NOK_POST;
		if ($wordLength > $wordMaxLength) return WS_LENGTH_NOK_WORD;
	}
	return WS_LENGTH_OK;
}

/*
 * Returns output text for a tokens list.
 * $tokens : tokens list.
 * $out : output format.
 */
function &WS_getOutput(&$tokens, $out)
{
	global $WS_Visitors;
	$output = '';
	foreach($tokens as $token) {
		$output .= $token->tokenType->accept($token, $WS_Visitors[$out]);
	}
	return $output;
}

/*
 * Serialize a token (list)* according to rfcc39.
 */
function &WS_serialize(&$tokens)
{
	global $WS_InitSymbTab;
	$serializer = new ALE_Serializer($WS_InitSymbTab);
	return $serializer->serialize($tokens);
}

/*
 * Deserialize a token (list)* according to rfcc39.
 */
function &WS_unserialize(&$tokens)
{
	global $WS_TokenTypes, $WS_InitSymbTab;
	$unserializer = new ALE_Unserializer($WS_TokenTypes, $tokens,
		$WS_InitSymbTab);
	return $unserializer->unserialize();
}

/*
 * Returns an equivalent token list where successive textual tokens were
 * merged (according to their relevant text).
 */
function &WS_mergeTextTokens(&$tokens)
{
	global $WS_T_Blank, $WS_T_Text;
	if (count($tokens) == 0) return array();
	$newTokens = array();
	$last = 0;
	
	if ($tokens[0]->tokenType->isText($tokens[0])) {
		$newTokens[0] = new ALE_Token($WS_T_Text, $tokens[0]->relevantText());
	} else {
		$newTokens[0] = $tokens[0];
	}
	
	for ($i = 1; $i < count($tokens); $i++) {
		if ($newTokens[$last]->tokenType == $WS_T_Text) {
			if ($tokens[$i]->tokenType->isText($tokens[$i])) {
				$newTokens[$last]->text .= $tokens[$i]->relevantText();
			} else {
				array_push($newTokens, $tokens[$i]);
				$last++;
			}
		} else {
			if ($tokens[$i]->tokenType->isText($tokens[$i])) {
				array_push($newTokens,
					new ALE_Token($WS_T_Text, $tokens[$i]->relevantText()));
			} else {
				array_push($newTokens, $tokens[$i]);
			}
			$last++;
		}
	}
	
	return $newTokens;
}

?>
