<?php
/*
 * chouchenn wonderslip
 * Copyright (c) 2004 olo <wonderslip at olivierl dot org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/**************************************************
 * Token types
 **************************************************/

/*
 * Token type, used for "visitor" design pattern adaptation, and to get the
 * length of a token.
 */
class WS_TokenType extends ALE_TokenType
{
	/*
	 * Calls the appropriate method of a visitor, depending on the token type
	 * name.
	 * $token : token to pass on the visitor.
	 * $visitor : visitor to call.
	 */
	function accept(&$token, &$visitor)
	{
		return $visitor->{"visitToken_$this->name"}($token);
	}
	
	/*
	 * Returns the XML-compliant text of a token.
	 */
	function &xmlText(&$token)
	{
		return htmlspecialchars($token->relevantText());
	}
	
	/*
	 * Tells if a token is a textual one (for token merging).
	 */
	function isText(&$token) { return false; }
	
	/*
	 * Tells if a token is a space (for length checking).
	 */
	function isSpace(&$token) { return false; }
}

/*
 * End of data.
 * (no properties)
 */
class WS_TokenType_End extends WS_TokenType
{
	function WS_TokenType_End() { $this->name = 'end'; }
	function &relevantText(&$token) { return ''; }
	function &relevantProperties(&$token) { return array(); }
}

/*
 * Simple text.
 * (no properties)
 */
class WS_TokenType_Text extends WS_TokenType
{
	function WS_TokenType_Text() { $this->name = 'text'; }
	function &relevantProperties(&$token) { return array(); }
	function isText(&$token) { return true; }
}

/*
 * Blank characters sequence.
 * (no properties)
 */
class WS_TokenType_Blank extends WS_TokenType
{
	function WS_TokenType_Blank() { $this->name = 'blank'; }
	function &relevantText(&$token) { return ' '; }
	function &relevantProperties(&$token) { return array(); }
	function isText(&$token) { return true; }
	function isSpace(&$token) { return true; }
}

/*
 * URL.
 * properties :
 * - url et title (optional)
 */
class WS_TokenType_Url extends WS_TokenType
{
	function WS_TokenType_Url() { $this->name = 'url'; }
	function &relevantText(&$token)
	{
		return (isset($token->properties['url']))
			? $token->properties['title']
			: $token->text;
	}
	function &relevantProperties(&$token)
	{
		$prop = array();
		if (isset($token->properties['url']))
		{
			$prop['url'] = $token->properties['url'];
			$prop['title'] = $token->properties['title'];
		}
		return $prop;
	}
}

/*
 * Wiki link.
 * properties :
 * - keyword
 * - name (wiki name) OR site (wiki base url) (optional)
 */
class WS_TokenType_Wiki extends WS_TokenType
{
	function WS_TokenType_Wiki() { $this->name = 'wiki'; }
	function &relevantText(&$token) { return ''; }
	function &relevantProperties(&$token)
	{
		$prop = array('keyword' => $token->properties['keyword']);
		if (isset($token->properties['site']))
		{
			$prop['site'] = $token->properties['site'];
		}
		else if (isset($token->properties['name']))
		{
			$prop['name'] = $token->properties['name'];
		}
		return $prop;
	}
	function &xmlText(&$token)
	{
		$text = '['.htmlspecialchars($token->properties['keyword']);
		if (isset($token->properties['site']))
		{
			$text .= '@'.$token->properties['site'];
		}
		else if (isset($token->properties['name']))
		{
			$text .= '@'.$token->properties['name'];
		}
		$text .= ']';
		return $text;
	}
}

/*
 * Smiley.
 * properties :
 * - name
 */
class WS_TokenType_Smiley extends WS_TokenType
{
	function WS_TokenType_Smiley() { $this->name = 'smiley'; }
	function &relevantText(&$token) { return ''; }
	function &relevantProperties(&$token)
	{
		return array('name' => $token->properties['name']);
	}
	function &xmlText(&$token)
	{
		return '[:'.htmlspecialchars($token->properties['name']).']';
	}
}

/*
 * Opening tag.
 * properties :
 * - confirmed
 * - name
 */
class WS_TokenType_BeginTag extends WS_TokenType
{
	function WS_TokenType_BeginTag() { $this->name = 'begintag'; }
	function &relevantText(&$token)
	{
		return $token->properties['confirmed']
			? ''
			: '<'.$token->properties['name'].'>';
	}
	function &relevantProperties(&$token)
	{
		return $token->properties['confirmed']
			? array('confirmed' => true, 'name' => $token->properties['name'])
			: array('confirmed' => false);
	}
	function isText(&$token) { return !$token->properties['confirmed']; }
}

/*
 * Closing tag.
 * properties :
 * - confirmed
 * - name
 */
class WS_TokenType_EndTag extends WS_TokenType
{
	function WS_TokenType_EndTag() { $this->name = 'endtag'; }
	function &relevantText(&$token)
	{
		return $token->properties['confirmed']
			? ''
			: '</'.$token->properties['name'].'>';
	}
	function &relevantProperties(&$token)
	{
		return $token->properties['confirmed']
			? array('confirmed' => true, 'name' => $token->properties['name'])
			: array('confirmed' => false);
	}
	function isText(&$token) { return !$token->properties['confirmed']; }
}

/*
 * Clock.
 * properties :
 * - valid
 * - hours
 * - minutes
 * - seconds (optional)
 * - index (optional)
 * - board (optional)
 * - sub (optional)
 */
class WS_TokenType_Clock extends WS_TokenType
{
	function WS_TokenType_Clock() { $this->name = 'clock'; }
	function &relevantText(&$token) { return $token->text; }
	function &relevantProperties(&$token)
	{
		if (!$token->properties['valid']) return array('valid' => false);
		$prop = array(
			'valid' => true,
			'hours' => $token->properties['hours'],
			'minutes' => $token->properties['minutes']
		);
		if (isset($token->properties['seconds']))
		{
			$prop['seconds'] = $token->properties['seconds'];
		}
		if (isset($token->properties['index']))
		{
			$prop['index'] = $token->properties['index'];
		}
		if (isset($token->properties['board']))
		{
			$prop['board'] = $token->properties['board'];
		}
		if (isset($token->properties['sub']))
		{
			$prop['sub'] = $token->properties['sub'];
		}
		return $prop;
	}
	function isText(&$token) { return !$token->properties['valid']; }
}

$WS_T_End = new WS_TokenType_End();
$WS_T_Blank = new WS_TokenType_Blank();
$WS_T_Smiley = new WS_TokenType_Smiley();
$WS_T_Wiki = new WS_TokenType_Wiki();
$WS_T_BeginTag = new WS_TokenType_BeginTag();
$WS_T_EndTag = new WS_TokenType_EndTag();
$WS_T_Text = new WS_TokenType_Text();
$WS_T_Url = new WS_TokenType_Url();
$WS_T_Clock = new WS_TokenType_Clock();

/*
 * Token types associative array (name -> tokenType), for deserialization.
 */
$WS_TokenTypes = array(
	'end' => $WS_T_End,
	'text' => $WS_T_Text,
	'blank' => $WS_T_Blank,
	'url' => $WS_T_Url,
	'wiki' => $WS_T_Wiki,
	'smiley' => $WS_T_Smiley,
	'begintag' => $WS_T_BeginTag,
	'endtag' => $WS_T_EndTag,
	'clock' => $WS_T_Clock
);

/*
 * Initial symbols table for tokenization.
 */
$WS_InitSymbTab = array(
	0 => 'text',
	1 => 'blank',
	2 => 'url',
	3 => 'wiki',
	4 => 'smiley',
	5 => 'begintag',
	6 => 'endtag',
	7 => 'clock',
	8 => 'title',
	9 => 'keyword',
	10 => 'site',
	11 => 'name',
	12 => 'confirmed',
	13 => 'valid',
	14 => 'hours',
	15 => 'minutes',
	16 => 'seconds',
	17 => 'index',
	18 => 'board',
	19 => 'sub'
);

?>
