<?php
/*
 * chouchenn wonderslip
 * Copyright (c) 2004 olo <wonderslip at olivierl dot org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/**************************************************
 * Token types
 **************************************************/

/*
 * Token type, used for "visitor" design pattern adaptation, and to get the
 * length of a token.
 */
class WS_TokenType extends ALE_TokenType
{
	var $length;
	var $isText;
	
	/*
	 * Creates a new token type.
	 * $name : (unique) name of the token type.
	 * $length : if strictly positive : the fixed size of the token, else :
	 * number of characters to add to the token text length to get the actual
	 * token size.
	 * $isText : tells if the token must be considered as (part of) a word.
	 */
	function WS_TokenType($name, $length = 0, $isText = false)
	{
		$this->name = $name;
		$this->length = $length;
		$this->isText = $isText;
	}
	
	/*
	 * Calls the appropriate method of a visitor, depending on the token type
	 * name.
	 * $token : token to pass on the visitor.
	 * $visitor : visitor to call.
	 */
	function accept(&$token, &$visitor)
	{
		return $visitor->{"visitToken_$this->name"}($token);
	}
	
	/*
	 * Returns the length of a token.
	 */
	function getLength(&$token)
	{
		return array(
			$this->length > 0
				? $this->length
				: strlen($token->string) + $this->length,
			$this->isText
		);
	}
}

/*
 * End of data.
 * (no properties)
 */
$WS_T_End = new WS_TokenType('end');

/*
 * Blank characters sequence.
 * (no properties)
 */
$WS_T_Blank = new WS_TokenType('blank', 1);

/*
 * Smiley.
 * properties :
 * - name
 */
$WS_T_Smiley = new WS_TokenType('smiley');

/*
 * Wiki link.
 * properties :
 * - keyword
 * - name (wiki name) or site (wiki base url) (optional)
 */
$WS_T_Wiki = new WS_TokenType('wiki');

/*
 * Opening tag.
 * properties :
 * - confirmed
 * - name
 */
$WS_T_BeginTag = new WS_TokenType('begintag');

/*
 * Closing tag.
 * properties :
 * - confirmed
 * - name
 */
$WS_T_EndTag = new WS_TokenType('endtag');

/*
 * Simple text.
 * (no properties)
 */
$WS_T_Text = new WS_TokenType('text', 0, true);

/*
 * URL.
 * properties :
 * - url (optional)
 * - title (optional)
 */
$WS_T_Url = new WS_TokenType('url');

/*
 * Clock.
 * properties :
 * - valid
 * - hours
 * - minutes
 * - seconds (optional)
 * - index (optional)
 * - board (optional)
 * - sub (optional)
 */
$WS_T_Clock = new WS_TokenType('clock', 0, true);

/*
 * Token types associative array (name -> tokenType), for deserialization.
 */
$WS_TokenTypes = array(
	'end' => $WS_T_End,
	'blank' => $WS_T_Blank,
	'smiley' => $WS_T_Smiley,
	'wiki' => $WS_T_Wiki,
	'begintag' => $WS_T_BeginTag,
	'endtag' => $WS_T_EndTag,
	'text' => $WS_T_Text,
	'url' => $WS_T_Url,
	'clock' => $WS_T_Clock
);

?>
