<?php
/*
 * chouchenn wonderslip
 * Copyright (c) 2004 olo <wonderslip at olivierl dot org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/**************************************************
 * Automaton lexer engine - Transitions
 **************************************************/

/**abstract**/ class ALE_Transition
{
	/* Target state name. */
	var $targetState;
	
	/* Number of character consumed by the transition application. */
	var $consumption;
	
	/* Sequence of actions to execute juste before characters
	   consumption. */
	var $actions;

	/*
	 * Determines if the current transition is applicable within the
	 * context.
	 * $context : context.
	 * returns true if the transition is applicable.
	 */
	/**abstract**/ function isApplicableTo(&$context) {}
}

/* Alternative transition. */
class ALE_AlternativeTransition extends ALE_Transition
{
	/*
	 * Create an alternative transition, always applicable.
	 * $targetState : target state name.
	 * $consumption : number of character consumed by the transition
	 * application.
	 * $actions : sequence of actions to execute juste before characters
	 * consumption.
	 */
	function ALE_AlternativeTransition($targetState, $consumption = 0,
		$actions = array())
	{
		$this->targetState = $targetState;
		$this->consumption = $consumption;
		$this->actions =& $actions;
	}
	
	function isApplicableTo(&$context)
	{
		return true;
	}
}

/* Final transition. */
class ALE_EndTransition extends ALE_Transition
{
	/*
	 * Create a final transition, applicable when the end of data is reached.
	 * $targetState : target state name.
	 * $consumption : number of character consumed by the transition
	 * application.
	 * $actions : sequence of actions to execute juste before characters
	 * consumption.
	 */
	function ALE_EndTransition($targetState, $consumption = 0,
		$actions = array())
	{
		$this->targetState = $targetState;
		$this->consumption = $consumption;
		$this->actions =& $actions;
	}
	
	function isApplicableTo(&$context)
	{
		return $context->endOfData;
	}
}

/* "character" transition. */
class ALE_CharTransition extends ALE_Transition
{
	var $charClass;
	
	/*
	 * Create a "character" transition, applicable when current character
	 * belongs to a given class.
	 * $charClass : character class ("preg" regex form, '\w' or '[\d\.-]',
	 * for example).
	 * $targetState : target state name.
	 * $consumption : number of character consumed by the transition
	 * application.
	 * $actions : sequence of actions to execute juste before characters
	 * consumption.
	 */
	function ALE_CharTransition($charClass, $targetState, $consumption = 1,
		$actions = array())
	{
		$this->charClass = preg_replace(',#,', '\#', $charClass);
		$this->targetState = $targetState;
		$this->consumption = $consumption;
		$this->actions =& $actions;
	}
	
	function isApplicableTo(&$context)
	{
		return preg_match('#'.$this->charClass.'#', $context->currentChar);
	}
}

/* "regex" transition. */
class ALE_RegexTransition extends ALE_Transition
{
	var $pattern;
	
	/*
	 * Create a "regex" transition, applicable when current string matches a
	 * given pattern.
	 * $pattern : pattern to match ("preg" regex form, '#^(coin|pan)$#', for
	 * example).
	 * $targetState : target state name.
	 * $consumption : number of character consumed by the transition
	 * application.
	 * $actions : sequence of actions to execute juste before characters
	 * consumption.
	 */
	function ALE_RegexTransition($pattern, $targetState, $consumption = 1,
		$actions = array())
	{
		$this->pattern = $pattern;
		$this->targetState = $targetState;
		$this->consumption = $consumption;
		$this->actions =& $actions;
	}
	
	
	function isApplicableTo(&$context)
	{
		return preg_match($this->pattern,
			$context->currentString.$context->currentChar);
	}
}

?>
